package oracle.i18n.demo.obe.gdkshop.gdk;

import oracle.i18n.servlet.localesource.LocaleSource;
import oracle.i18n.demo.obe.gdkshop.ShopAction;
import oracle.i18n.servlet.ApplicationContext;
import oracle.i18n.servlet.ServletHelper;
import oracle.i18n.util.OraLocaleInfo;
import oracle.i18n.util.LocaleMapper;
import java.util.Locale;
import java.util.Map;
import java.util.HashMap;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;

import java.util.TimeZone;

import javax.servlet.http.HttpServletRequest;


/**
 * CustomDBLocaleSource is a database LocaleSource class that accesses the user
 * locale information on the database. <code>load</code> method fetches  the
 * user locale preference from the database, while <code>store</code>  method
 * updates the information.
 *
 * Default caching : This locale source is cached in the session based on userid
 */
public class CustomDBLocaleSource extends LocaleSource
{
  /** Constant for key to retrieve m_localeMap object */
  private static final String LOCALE_MAP_KEY = "oracle.i18n.demo.obe.gdkshop.gdk.CustomDBLocaleSource.m_localeMap";
 
  private String m_user;
  private ApplicationContext m_appctx;


  /**
   * Constructs this object
   *
   * @param user User object
   * @param appctx GDK/OGS ApplicationContext
   *
   * @throws SQLException if SQL error occurs
   */
  private CustomDBLocaleSource(String userid, ApplicationContext appctx)
    throws SQLException
  {
    super(true, // cache-able
          false // read-only
    );
    m_user = userid;
    m_appctx = appctx;
  }

  /**
   * Returns a LocaleSource instance
   *
   * @param request HttpServletRequest object
   *
   * @return LocaleSource a LocaleSource object
   */
  public static final LocaleSource getInstance(HttpServletRequest request)
  {
    // get authenticated user object
    String userid = (String) request.getSession().getAttribute("userid");
    if (userid == null)
       return null;
    try
    {
        Map localeMap = (Map)request.getSession().getAttribute(LOCALE_MAP_KEY);
        if (localeMap != null){
            LocaleSource locsrc = (LocaleSource)localeMap.get(userid);
            if (locsrc != null){
                return locsrc;
            }
        }else{   
            localeMap = new HashMap();
        }
        LocaleSource locsrc = new CustomDBLocaleSource(userid, 
          ServletHelper.getApplicationContextInstance(request));
        locsrc.load();
        localeMap.put(userid, locsrc);
        if (locsrc.isCached()){ 
           request.getSession().setAttribute(LOCALE_MAP_KEY, localeMap);
        }
        return locsrc;     
    }
    catch(Exception e)
    {
      throw new IllegalStateException(e.getMessage());
    }
  }

  /**
   * Returns a true if LocaleSource is accessible
   *
   * @param request HttpServletRequest object
   *
   * @return boolean 
   */
  public static boolean isAccessible(HttpServletRequest request)
  {
    String userid = (String) request.getSession().getAttribute("userid");

    return userid != null;
  }

  /**
   * Fetches all locale information from DB.
   *
   * @return true if the operation is success, otherwise false
   */ 
  public boolean load()
  {
    Connection cn = null;
    PreparedStatement ps = null;
    ResultSet rs = null;

    try
    {
      cn = ShopAction.getConnection();
      ps = cn.prepareStatement(
          "SELECT c.nls_language, c.nls_territory, ex.timezone "
          + "FROM customers c, ext_user_profile ex "
          + "WHERE c.customer_id = ex.customer_id AND "
          + "LOWER(c.cust_email) = LOWER(?)");
      ps.setString(1, m_user);
      rs = ps.executeQuery();
      rs.next();
      Locale locale = OraLocaleInfo.getInstance(
        LocaleMapper.getOraLanguageFromShortName(rs.getString(1)),
        rs.getString(2)).getLocale();
      locale = m_appctx.fallbackLocale(locale);
      setLocale(locale);
      setTimeZone(TimeZone.getTimeZone(rs.getString(3)));
    }
    catch (SQLException e)
    {
      m_user = null;

      return false;
    }
    ShopAction.closeAllObjects(cn, ps, rs);
    return true;
  }

  /**
   * Stores all locale information into the db.
   *
   * @return true if the operation is success, otherwise false
   */
  public boolean store()
  {
    Connection cn = null;
    PreparedStatement ps = null;

    try
    {
      cn = ShopAction.getConnection();
      ps = cn.prepareStatement(
          "UPDATE customers c, ext_user_profile ex "
          + "SET c.nls_language = ?, c.nls_territory = ?, ex.timezone =? "
          + "WHERE LOWER(c.cust_email) = LOWER(?) AND c.customer_id = ex.customer_id");
      OraLocaleInfo oloc = OraLocaleInfo.getInstance(getLocale());
      
      ps.setString(1, oloc.getShortLanguage().toLowerCase(Locale.US));
      ps.setString(2, oloc.getTerritory().toUpperCase(Locale.US));
      ps.setString(3, getTimeZone().getID());
      ps.setString(4, m_user);
      ps.execute();
    }
    catch (SQLException e)
    {
      m_user = null;

      return false;
    }
    ShopAction.closeAllObjects(cn, ps, null);
    return true;
  }
}
